/*+ NmeaParser.cpp
 *
 ******************************************************************************
 *
 *                        Trimble Navigation Limited
 *                           645 North Mary Avenue
 *                              P.O. Box 3642
 *                         Sunnyvale, CA 94088-3642
 *
 ******************************************************************************
 *
 *    Copyright  2007 Trimble Navigation Ltd.
 *    All Rights Reserved
 *
 ******************************************************************************
 *
 * Description:
 *    This file implements the CNmeaParser class.
 *            
 * Revision History:
 *    01-11-2007    Jacob Norda
 *                  Written
 *
 * Notes:
 *
-*/


/*---------------------------------------------------------------------------*\
 |                         I N C L U D E   F I L E S
\*---------------------------------------------------------------------------*/
#include "stdafx.h"
#include "nmeaparser.h"
#include "CommPort.h"

/*---------------------------------------------------------------------------*\
 |                  C O N S T A N T S   A N D   M A C R O S
\*---------------------------------------------------------------------------*/
#define MAX_NMEA_FIELD_SIZE  25



/*---------------------------------------------------------------------------*\
 |                 N M E A   P R O C E S S O R   R O U T I N E S 
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       IsStartOfNmea

Description:    Determines if the supplied byte could be the start of a new 
				NMEA packet, i.e. if it is a '$'. 

Parameters:     ucByte		 - a byte to test.

Return Value:   True if the byte could be the start of a new NMEA packet, 
				false otherwise. 
-----------------------------------------------------------------------------*/
bool CNmeaParser::IsStartOfNmea(U8 ucByte)
{
	return NMEA_START == ucByte; 
}

/*-----------------------------------------------------------------------------
Function:       Reset

Description:    Resets the parsing state machine to its initial state. 

                NOTE: This function must be called prior to the first call of
				ReceiveByte when receiving a new NMEA packet. 

Parameters:     None

Return Value:   None 
-----------------------------------------------------------------------------*/
void CNmeaParser::Reset()
{
	m_nParseState = MSG_IN_COMPLETE; 
}


/*-----------------------------------------------------------------------------
Function:       ReceiveByte

Description:    Receives a complete NMEA packet one byte at a time. Call the
				function repeatedly until it indicates that a full packet has
				been received (see Return Value below). 

                The entire packet (the starting $, packet header, packet data, 
                and trailing <cr><lf>) is placed in the buffer ucPkt. The 
                entire packet size is stored in pPktLen.

Parameters:     ucByte		 - The next byte to be added to the message. 
                ucPkt        - a memory buffer where the entire NMEA packet 
                               will be stored.
                pPktLen      - a pointer to a variable to be updated with the
                               packet size (which includes the packet header
                               and trailing bytes).

Return Value:   The internal parsing state:

				MSG_IN_ERROR: There was an error receiving the message. Reset()
				should be called and the input stream re-synched using 
				IsStartOfNmea().

				MSG_IN_COMPLETE: A complete message has been received in ucPkt
				and can be parsed by ParsePkt(). 

				Any other value: More bytes are needed to complete the packet, 
				keep feeding them in. 
-----------------------------------------------------------------------------*/
int CNmeaParser::ReceiveByte (U8 ucByte, U8 ucPkt[], int *pPktLen)
{
	// The TSIP packet is received in a local state machine.
	switch (m_nParseState) 
	{
	case MSG_IN_ERROR:		// Start over in case of error
	case MSG_IN_COMPLETE:               
		// This is the initial state in which we look for the start
		// of the NMEA packet. We can also end up in this state if
		// we received too many data bytes from the serial port but
		// did not find a valid NMEA packet in that data stream.
		// 
		// While in this state, we look for a '$' character. If we
		// are in this state and the '$' is received, we initialize
		// the packet buffer and transition to the next state.
		if (ucByte == NMEA_START) 
		{
			m_nParseState    = NMEA_IN_PARTIAL;
			*pPktLen          = 0;
			ucPkt[(*pPktLen)++] = ucByte;
		}
		break;

	case NMEA_IN_PARTIAL:

		/* Add the character to the current message. */
		ucPkt[(*pPktLen)++] = ucByte;

		/* 
		* Check for NMEA message terminator <cr> = NMEA_END_CR. All 
		* other characters are considered part of the message.
		*/
		if (ucByte == NMEA_END_CR) 
		{
			/* 
			* The first character of the <cr><lf> termination 
			* has been detected; we change the state and wait 
			* for the second.
			*/
			m_nParseState = NMEA_CR;
		}
		break;

	case NMEA_CR:
		/*
		* Check for NMEA message terminator <lf> = NMEA_END_LF. It
		* indicates a complete NMEA packet has been received.
		*/
		if (ucByte == NMEA_END_LF) 
		{
			ucPkt[(*pPktLen)++] = ucByte;
			m_nParseState = MSG_IN_COMPLETE;
		}
		else 
		{
			// This was a malformed NMEA message - return an error. 
			m_nParseState = MSG_IN_ERROR;
		}
		break;

	default:
		// We should never be in this state. This is just for a good
		// programming style.
		m_nParseState = MSG_IN_ERROR;
		break;
	}

	// We get to this point after reading each byte from the serial port.
	// Because it is not the end of the valid NMEA packet yet, we check for 
	// buffer overflow. If it overflows we assume something went wrong 
	// with the end of message character(s) since no input message should
	// be bigger than MAX_NMEA_PKT_LEN. We ignore this message and wait till 
	// the next message starts.                                    
	if (*pPktLen >= MAX_NMEA_PKT_LEN) 
	{
		m_nParseState = MSG_IN_ERROR;
		*pPktLen = 0;
	}

	return m_nParseState; 
}


/*-----------------------------------------------------------------------------
Function:       ParsePkt

Description:    This function extracts the data values from a NMEA packet
                and returns an ASCII-formatted string with the values.

Parameters:     ucPkt   - a memory buffer with the entire NMEA packet 
                nPktLen - size of the packet (including the header and 
                          trailing bytes).

Return Value:   A string with NMEA data values extracted and formatted as 
                ASCII text.
-----------------------------------------------------------------------------*/
CString CNmeaParser::ParsePkt   (U8 ucPkt[], int nPktLen)
{
	CString result;

	// NMEA packets end with <CR><LF>, but those characters are neither
	// needed nor useful for the message parsing. Null-terminate the 
	// string and update the packet length accordingly. 
	ucPkt[nPktLen-2] = '\0'; 
	nPktLen -= 2; 

	if (!CheckChecksum(ucPkt, nPktLen)) 
	{
		return "ERROR> Bad NMEA checksum"; 
	}

	// Default output in case packet isn't handled by the code below
	result.Format("Unsupported packet: %s", ucPkt);  

    // Process proprietary NMEA sentences (Trimble's or others')
    if (ucPkt[1] == 'P')
    {
        if (Matches(&ucPkt[2], "TNL"))
        {
            if ((ucPkt[6] == 'V') && (ucPkt[7] == 'R')) result = ParseVR (ucPkt, nPktLen);
			// Other packet handlers excluded for brevity
        }
    }

    // Process standard NMEA sentences
    if ((ucPkt[1] == 'G') && (ucPkt[2] == 'P'))
    {
        if (Matches(&ucPkt[3], "GGA")) result = ParseGGA (ucPkt, nPktLen);
		else if (Matches(&ucPkt[3], "VTG")) result = ParseVTG (ucPkt, nPktLen);
		// Other packet handlers excluded for brevity
    }

	return result; 
}


/*-----------------------------------------------------------------------------
Function:       SendPkt

Description:    This function sends a predetermined sample packet (VR) to the 
				supplied serial port. 

Parameters:     pPort   - The serial port to send the sample packet to.

Return Value:   None.
-----------------------------------------------------------------------------*/
void CNmeaParser::SendPkt(CCommPort* pPort)
{
	U8 ucPkt[MAX_NMEA_PKT_LEN]; 
	U16 nPktLen; 

	nPktLen = FormatVR(ucPkt); 

	pPort->Write(ucPkt, nPktLen); 
}


/*---------------------------------------------------------------------------*\
 |     I N D I V I D U A L   N M E A   P A C K E T   P A R S E R S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       ParseVR

Description:    Extracts the data values from the NMEA packet and returns the
                ASCII representations of the data values.

Parameters:     ucPkt	- a pointer to the start of the NMEA packet
                nPktLen - number of data bytes in ucPkt

Return Value:   none
-----------------------------------------------------------------------------*/
CString CNmeaParser::ParseVR  (U8 ucPkt[], int nPktLen)
{
	CString result; 
    S8 cField[MAX_NMEA_FIELD_SIZE];
	int offset;
	bool status; 
	U8 versionType; 
	S8 cName[MAX_NMEA_FIELD_SIZE]; 
	int major, minor, beta, month, day, year; 


	offset = 0; 
	// First field is the sentence header, ignore
	status = GetNextField(ucPkt, cField, &offset); 
	if (!status) return "Malformed VR sentence"; 

	status = GetNextFieldChar(ucPkt, cField, &offset, &versionType); 
	if (!status) return "Malformed VR sentence"; 

	status = GetNextField(ucPkt, cName, &offset); 
	if (!status) return "Malformed VR sentence"; 

	// Version field is xx.xx.xx - need to get full field 
	// in a string and then parse out the individual components
	status = GetNextField(ucPkt, cField, &offset);
	if (!status) return "Malformed VR sentence"; 

	major = atoi(cField); 
	minor = atoi(cField+3); 
	beta  = atoi(cField+6); 

	status = GetNextFieldInt(ucPkt, cField, &offset, &month); 
	if (!status) return "Malformed VR sentence"; 
	status = GetNextFieldInt(ucPkt, cField, &offset, &day); 
	if (!status) return "Malformed VR sentence"; 
	status = GetNextFieldInt(ucPkt, cField, &offset, &year); 
	if (!status) return "Malformed VR sentence"; 

	result.Format("VR: %c, %2d.%02d.%02d, %d/%d/%d, %s", 
		versionType, major, minor, beta, month, day, year, cName); 

	return result; 
}


/*-----------------------------------------------------------------------------
Function:       ParseGGA

Description:    Extracts the data values from the NMEA packet and returns the
                ASCII representations of the data values.

Parameters:     ucPkt	- a pointer to the start of the NMEA packet
                nPktLen - number of data bytes in ucPkt

Return Value:   none
-----------------------------------------------------------------------------*/
CString CNmeaParser::ParseGGA (U8 ucPkt[], int nPktLen)
{
	CString result; 
    S8 cField[MAX_NMEA_FIELD_SIZE];
	int offset;
	bool status; 

	U8 tmpChar; 
	int tmpInt, hour, minute, second, fixSource, numSvsUsed, dgpsAge, dgpsStationId; 
	DBL tmpDbl, lat, lon, hdop, alt, heightOfGeoid;
	U8 altUnit, geoidUnit; 


	offset = 0; 
	// First field is the sentence header, ignore
	status = GetNextField(ucPkt, cField, &offset); 
	if (!status) return "Malformed GGA sentence"; 

	// Time is encoded as HHMMSS in one int field
	status = GetNextFieldInt(ucPkt, cField, &offset, &tmpInt); 
	second = tmpInt % 100; tmpInt /= 100; 
	minute = tmpInt % 100; tmpInt /= 100; 
	hour = tmpInt; 


	status = GetNextFieldDbl(ucPkt, cField, &offset, &tmpDbl); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &tmpChar); 

	// lat & lon are in the format DDMM.MMM, convert to DD.DDD...
	tmpDbl /= 100.0;						 // DDMM.MMM -> DD.MMMMM
	lat = (tmpDbl - (U32)tmpDbl)*100.0/60.0; // .MMMMM -> .DDDDD 
	lat += (U32)tmpDbl;						 // DD + .DDDDD -> DD.DDDDD 
	if (tmpChar == 'S') lat = -lat;			 // South is negative


	status = GetNextFieldDbl(ucPkt, cField, &offset, &tmpDbl); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &tmpChar); 

	// lat & lon are in the format DDMM.MMMMM, convert to DD.DDD...
	tmpDbl /= 100.0;						 // DDMM.MMMMM -> DD.MMMMMMM
	lon = (tmpDbl - (U32)tmpDbl)*100.0/60.0; // .MMMMMMM -> .DDDDDDD 
	lon += (U32)tmpDbl;						 // DD + .DDDDDDD -> DD.DDDDDDD 
	if (tmpChar == 'W') lon = -lon;			 // West is negative
	
	status = GetNextFieldInt(ucPkt, cField, &offset, &fixSource); 
	status = GetNextFieldInt(ucPkt, cField, &offset, &numSvsUsed); 
	status = GetNextFieldDbl(ucPkt, cField, &offset, &hdop); 

	status = GetNextFieldDbl(ucPkt, cField, &offset, &alt); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &altUnit); 

	status = GetNextFieldDbl(ucPkt, cField, &offset, &heightOfGeoid); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &geoidUnit); 

	status = GetNextFieldInt(ucPkt, cField, &offset, &dgpsAge); 
	status = GetNextFieldInt(ucPkt, cField, &offset, &dgpsStationId); 

	// Don't check number of fields in packet as the NMEA standard allows extending 
	// packets with more fields. 

	result.Format("GGA: %2d:%02d:%02d, %s, %2d satellites hdop: %4.2f,\r\n  lat:%11.7f lon:%11.7f alt:%5.1f", 
		hour, minute, second, (fixSource ? "VALID FIX" : "INVALID FIX"), numSvsUsed, hdop, lat, lon, alt); 

	return result; 
}


/*-----------------------------------------------------------------------------
Function:       ParseVTG

Description:    Extracts the data values from the NMEA packet and returns the
                ASCII representations of the data values.

Parameters:     ucPkt	- a pointer to the start of the NMEA packet
                nPktLen - number of data bytes in ucPkt

Return Value:   none
-----------------------------------------------------------------------------*/
CString CNmeaParser::ParseVTG (U8 ucPkt[], int nPktLen)
{
	CString result; 
    S8 cField[MAX_NMEA_FIELD_SIZE];
	int offset;
	bool status; 
	U8 tmpChar, fixMode; 
	DBL courseTrue, courseMagnetic, speedKnots, speedKph; 

	offset = 0; 
	// First field is the sentence header, ignore
	status = GetNextField(ucPkt, cField, &offset); 
	if (!status) return "Malformed VTG sentence"; 

	status = GetNextFieldDbl(ucPkt, cField, &offset, &courseTrue); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &tmpChar); 

	status = GetNextFieldDbl(ucPkt, cField, &offset, &courseMagnetic); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &tmpChar); 

	status = GetNextFieldDbl(ucPkt, cField, &offset, &speedKnots); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &tmpChar); 

	status = GetNextFieldDbl(ucPkt, cField, &offset, &speedKph); 
	status = GetNextFieldChar(ucPkt, cField, &offset, &tmpChar); 

	status = GetNextFieldChar(ucPkt, cField, &offset, &fixMode); 
	if (!status) fixMode = 'U'; 

	// Don't check number of fields in packet as the NMEA standard allows extending 
	// packets with more fields. 

	result.Format("VTG: course, true: %.2f magnetic: %.2f;\r\n  speed: %.2f knots, %.2f km/h, fixMode: %c", 
		courseTrue, courseMagnetic, speedKnots, speedKph, fixMode); 

	return result; 
}


/*-----------------------------------------------------------------------------
Function:       FormatVR

Description:    Formats the NMEA packet into the supplied buffer (which is 
				assumed to be of length MAX_NMEA_PKT_LEN), returning the number
				of bytes actually used. 

Parameters:     ucPkt	- a pointer to the buffer to hold the formatted NMEA packet

Return Value:   The number of bytes written to ucPkt
-----------------------------------------------------------------------------*/
int CNmeaParser::FormatVR(U8 ucPkt[]) 
{
	int len; 

	FormatNmeaMsgHead(ucPkt, &len, 'Q', "VR"); 
	PutNextField(ucPkt, "N", &len); 
	FormatNmeaMsgTail(ucPkt, &len); 

	return len; 
}



/*---------------------------------------------------------------------------*\
 |            U T I L I T Y   R O U T I N E S
\*---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
Function:       ComputeChecksum

Description:    Computes the checksum of the supplied NMEA packet. It requires
				that the checksum-separating '*' be present in the packet, but 
				not that the actual checksum is present. This makes it possible 
				to use this function both for parsing and formatting of NMEA
				packets. 

Parameters:     ucPkt	- a pointer to the start of the NMEA packet
                nPktLen - number of data bytes in ucPkt

Return Value:   The computed checksum for the supplied packet. 
				-1 (0xFF) is returned if the packet is malformed. 
-----------------------------------------------------------------------------*/
U8 CNmeaParser::ComputeChecksum(U8 ucPkt[], int nPktLen)
{
	U8 ucCsum;
	int i;

	ucCsum = 0;

	// The '$' & '*' are not part of the checksum - it is done between them. 
	// Be flexible on the location of the '*' so that we can use this routine
	// both when decoding and when constructing packets. 
	for (i=1; i<nPktLen; i++) 
	{
		if (ucPkt[i] == '*')
			break;
		ucCsum ^= ucPkt[i];
	}

	if (i < nPktLen && ucPkt[i] == '*')
		return ucCsum; 
	else 
		return -1; // Limited error handling  
}

/*-----------------------------------------------------------------------------
Function:       CheckChecksum

Description:    Checks the checksum of the supplied NMEA packet. It requires
				that the checksum-separating '*' and that the actual checksum 
				is present.  

Parameters:     ucPkt	- a pointer to the start of the NMEA packet
                nPktLen - number of data bytes in ucPkt

Return Value:   True if the checksum is correct, 
				False otherwise. 
-----------------------------------------------------------------------------*/
bool CNmeaParser::CheckChecksum(U8 ucPkt[], int nPktLen)
{
	return ((U8)GetHex(&ucPkt[nPktLen-2], 2)) == ComputeChecksum(ucPkt, nPktLen); 
}


/*-----------------------------------------------------------------------------
Function:       Matches

Description:    Checks if the contents of pPkt and pRef match each other on a 
				byte-by-byte basis. Both are pointers to null-terminated 
				strings, but pPkt doesn't have to end where pRef does (i.e. 
				pRef can point to what is intended to be the start of pPkt). 
				
Parameters:     pPkt - a pointer to a null-terminated string (e.g. ABCDE\0)
                pRef - a pointer to a null-terminated string (e.g. AB\0)

Return Value:   True if the start of pPkt is the same as the full pRef, 
				False otherwise. 
-----------------------------------------------------------------------------*/
bool CNmeaParser::Matches(U8* pucPkt, const S8* pcRef) 
{
	while (*pucPkt != '\0' && *pcRef != '\0') 
	{ 
		if (*pucPkt++ != *pcRef++)
			return false; 
	}
	return *pcRef == '\0'; 
}



/*-----------------------------------------------------------------------------
Function:       GetHex

Description:    Reads n hexadecimal characters from pcStr and converts them to 
				the corresponding number. E.g. '1A4C' -> 6732. 
				
Parameters:     pucStr - A pointer to an array of hexadecimal characters. 
                n      - The number of characters to read. 

Return Value:   The number the hexadecimal characters represent. 
-----------------------------------------------------------------------------*/
U32 CNmeaParser::GetHex (U8* pucStr, U16 n)
{
    U32 sum;
    U16 i;
	U8  h; 

    sum = 0;

    for (i=0; i<n; i++) 
    {
		if (pucStr[i] >= '0' && pucStr[i] <= '9') 
		{
			h = pucStr[i] - '0'; 
		} 
		else if (pucStr[i] >= 'A' && pucStr[i] <= 'F') 
		{
			h = pucStr[i] - 'A' + 10; 
		} 
		else if (pucStr[i] >= 'a' && pucStr[i] <= 'f') 
		{
			h = pucStr[i] - 'a' + 10; 
		} 
		else 
		{
			// Limited error handling - just return a zero. 
			// Have to make sure this doesn't map to the value
			// returned by ComputeChecksum in case of a failed checksum. 
			// All other data should have been pre-validated. 
			return 0; 
		}
        sum = 16*sum + h;
    }

    return (sum);
}


/*-----------------------------------------------------------------------------
Function:       GetNextField

Description:    Extracts the next field from the NMEA sentence and puts it in 
				the buffer pointed to by pcField. 

				NOTE: The buffer pcField points to must contain at least 
				MAX_NMEA_FIELD_SIZE bytes. 

				NOTE: pOffset is updated to point to the start of the next field
				when this function returns, so you can call it repeatedly to 
				extract all the fields in an NMEA packet.

Parameters:     pucStr       - pointer to an NMEA packet
                pcField      - a buffer where the extracted field will be copied
                pOffset      - the starting index in pucStr of the next field

Return Value:   true if field was extracted successfully; false otherwise
-----------------------------------------------------------------------------*/
bool CNmeaParser::GetNextField (U8* pucStr, S8* pcField, int* pOffset)
{
	S8* pcStr = (S8*)pucStr; 
	bool result; 

	if ((pcStr == NULL) || (pcField == NULL) || (pOffset == NULL))
	{
		return false;
	}

	// Copy field from pcStr to pcField
	int i = *pOffset;
	int i2 = 0;

	while ((pcStr[i] != ',') && (pcStr[i] != '*') && pcStr[i])
	{
		pcField[i2] = pcStr[i];
		i2++; i++;

		if (i2 >= MAX_NMEA_FIELD_SIZE)
		{
			i2 = MAX_NMEA_FIELD_SIZE-1;
			break;
		}
	}
	pcField[i2] = '\0';

	// Return true if the parsed field is not empty. This is the case if i incremented 
	// in the while-loop, i.e. if i and *pOffset are no longer the same. Otherwise, the 
	// first character in the field was the end character (, or *).
	result = i != *pOffset;

	// Be sure to step over the separator (, or *), i.e. need to go one more step than 
	// the loop above. However, don't do this if we have exhausted the source string
	// since this means that there are no more fields to parse. 
	if (pcStr[i]) {
		// There are characters left in the source string
		*pOffset = i + 1; 
	} else {
		// We've reached the end of the source string, don't step further
		*pOffset = i; 
	}

	return result;
}


/*-----------------------------------------------------------------------------
Function:       GetNextFieldChar

Description:    Customized wrapper function to GetNextField() that extracts 
				interprets the next field as a char. 

				NOTE: The buffer pcField points to must contain at least 
				MAX_NMEA_FIELD_SIZE bytes. 

				NOTE: pOffset is updated to point to the start of the next field
				when this function returns, so you can call it repeatedly to 
				extract all the fields in an NMEA packet.

Parameters:     pucStr   - pointer to an NMEA packet
                pcField  - a buffer where the extracted field will be copied
                pOffset  - the starting index in pucStr of the next field
				pChar	 - Pointer to the variable which is to receive the 
						   decoded value.  

Return Value:   true if field was extracted successfully; false otherwise
-----------------------------------------------------------------------------*/
bool CNmeaParser::GetNextFieldChar (U8* pucStr, S8* pcField, int* pOffset, U8* pChar)
{
	if (GetNextField(pucStr, pcField, pOffset))
	{ 
		*pChar = pcField[0]; 
		return true; 
	} 
	else 
	{
		*pChar = 0;
		return false; 
	}
}

/*-----------------------------------------------------------------------------
Function:       GetNextFieldInt

Description:    Customized wrapper function to GetNextField() that extracts 
				interprets the next field as an int. 

				NOTE: The buffer pcField points to must contain at least 
				MAX_NMEA_FIELD_SIZE bytes. 

				NOTE: pOffset is updated to point to the start of the next field
				when this function returns, so you can call it repeatedly to 
				extract all the fields in an NMEA packet.

Parameters:     pucStr   - pointer to an NMEA packet
                pcField  - a buffer where the extracted field will be copied
                pOffset  - the starting index in pucStr of the next field
				pInt	 - Pointer to the variable which is to receive the 
						   decoded value.  

Return Value:   true if field was extracted successfully; false otherwise
-----------------------------------------------------------------------------*/
bool CNmeaParser::GetNextFieldInt (U8* pucStr, S8* pcField, int* pOffset, int* pInt)
{
	if (GetNextField(pucStr, pcField, pOffset))
	{ 
		*pInt = atoi(pcField); 
		return true; 
	} 
	else 
	{
		*pInt = 0;
		return false; 
	}
}

/*-----------------------------------------------------------------------------
Function:       GetNextFieldDbl

Description:    Customized wrapper function to GetNextField() that extracts 
				interprets the next field as a double. 

				NOTE: The buffer pcField points to must contain at least 
				MAX_NMEA_FIELD_SIZE bytes. 

				NOTE: pOffset is updated to point to the start of the next field
				when this function returns, so you can call it repeatedly to 
				extract all the fields in an NMEA packet.

Parameters:     pucStr   - pointer to an NMEA packet
                pcField  - a buffer where the extracted field will be copied
                pOffset  - the starting index in pucStr of the next field
				pDbl	 - Pointer to the variable which is to receive the 
						   decoded value.  

Return Value:   true if field was extracted successfully; false otherwise
-----------------------------------------------------------------------------*/
bool CNmeaParser::GetNextFieldDbl (U8* pucStr, S8* pcField, int* pOffset, DBL* pDbl)
{
	if (GetNextField(pucStr, pcField, pOffset))
	{ 
		*pDbl = atof(pcField); 
		return true; 
	} 
	else 
	{
		*pDbl = 0.0;
		return false; 
	}
}



/*-----------------------------------------------------------------------------
Function:       FormatNmeaMsgHead

Description:    Formats the NMEA packet header into the supplied buffer (which is 
				assumed to be of length MAX_NMEA_PKT_LEN), updating pPktLen to 
				contain the number of bytes actually used by the header. 

Parameters:     pucStr	- a pointer to the buffer to hold the formatted NMEA packet
				pPktLen	- a pointer to the int to hold the number of bytes actually
						  occupied by the header
				ucMode	- a char indicating the mode to use - (S)et or (Q)uery
				pMsg	- a pointer to the NMEA message mnemonic to use, e.g. "VR" 

Return Value:   None
-----------------------------------------------------------------------------*/
void CNmeaParser::FormatNmeaMsgHead (U8* pucStr, int* pPktLen, U8 ucMode, S8* pMsg)
{
	int i = 0; 
	pucStr[i++] = '$';

	pucStr[i++] = 'P'; 
	pucStr[i++] = 'T'; 
	pucStr[i++] = 'N'; 
	pucStr[i++] = 'L'; 

	// Set the mode parameter, (S)et or (Q)uery
	pucStr[i++] = ucMode; 

	while (*pMsg != '\0') {
		pucStr[i++] = *pMsg++; 
	}
	
	*pPktLen = i; 
}

/*-----------------------------------------------------------------------------
Function:       FormatNmeaMsgTail

Description:    Formats the NMEA packet tail into the supplied buffer (which is 
				assumed to be of length MAX_NMEA_PKT_LEN), updating pPktLen to 
				contain the number of bytes actually used by the full packet. 
				
				The NMEA tail contains the '*', checksum and <cr><lf>. 

Parameters:     pucStr	- a pointer to the buffer to hold the formatted NMEA packet
				pPktLen	- a pointer to the int to hold the number of bytes actually
						  occupied by the packet

Return Value:   None
-----------------------------------------------------------------------------*/
void CNmeaParser::FormatNmeaMsgTail (U8* pucStr, int* pPktLen)
{
	int i = *pPktLen; 
	U8 checksum; 

	pucStr[i++] = '*'; 
	checksum = ComputeChecksum(pucStr, i); 
	pucStr[i++] = DigitToHex(checksum >> 4); 
	pucStr[i++] = DigitToHex(checksum); 
	pucStr[i++] = NMEA_END_CR; 
	pucStr[i++] = NMEA_END_LF; 

	*pPktLen = i; 
}

/*-----------------------------------------------------------------------------
Function:       DigitToHex

Description:    Converts a digit in the 0-15 range to ['0'-'9','A'-'F']
				
				NOTE: The supplied digit is truncated to be within 0-15. 

Parameters:     ucDigit	- the digit to convert

Return Value:   The hexadecimal character representing the supplied digit. 
-----------------------------------------------------------------------------*/
U8   CNmeaParser::DigitToHex(U8 ucDigit)
{
	ucDigit &= 0x0f; 
	if (ucDigit < 10) { 
		return ucDigit + '0'; 
	} else {
		return ucDigit + 'A' - 10; 
	}
}

/*-----------------------------------------------------------------------------
Function:       PutNextField

Description:    Appends the null-terminated string in pcField to pucStr, 
				starting from pOffset and prepending an NMEA separator (','). 

				NOTE: The buffer pucStr points to must contain at least 
				MAX_NMEA_PKT_LEN bytes. 

				NOTE: pOffset is updated to point to the start of the next field
				when this function returns, so you can call it repeatedly to 
				format all the fields in an NMEA packet.

Parameters:     pucStr       - pointer to the NMEA packet which is to have a 
							   field added to it. 
                pcField      - the field to append (a null-terminated string)
                pOffset      - the starting index in pucStr of the next field

Return Value:   None
-----------------------------------------------------------------------------*/
void CNmeaParser::PutNextField(U8* pucStr, S8* pcField, int* pOffset)
{
	S8* pcStr = (S8*)&pucStr[*pOffset]; 
	int i; 

	// Put the separator
	*pcStr++ = ','; 

	// Put the actual field - pcField must be null-terminated! 
	for (i=0; pcField[i] != '\0'; i++) 
	{ 
		pcStr[i] = pcField[i]; 
	}

	// Update the offset, taking the separator into account
	*pOffset += i + 1; 
}



